<?PHP if ( ! defined('BASEPATH')) exit('No direct script access allowed');

class UsersModel extends CI_Model {
	
    function __construct() {
        parent::__construct();
		$this->load->database();
    }
    
	/* This function will return an associative array with the SQL result and LDAP result of the user
	 * specified by the user id (database index) parameter
	 */
	public function get_user($user_id) {
		$username = $this->get_username_from_id($user_id); 
		if($username) { 
			$result = $this->get_user_from_username($username);
			if($result) { 
				$result->disabled = FALSE;
				return $result; 
			}
			else {
				//if it's not an active account, check if it is disabled
				$result = $this->get_user_from_username($username,FALSE);
				if($result) {
					$result->disabled = TRUE;
					return $result;
				}
			}
		}
		return FALSE;
	}
	
	public function get_users() {
		$users = $this->get_user_from_username('*');
		if(!is_array($users)) { $users = array($users); }
		return $users;
	}
	
	public function get_users_by_page_number($start, $size) {
		$users_by_page_number = array();
		$users = $this->get_users();
		$j = 0;
		for($i = 0; $i < sizeOf($users); $i++) {
			
			if($i >= $start - 1 && $i < $start - 1 + $size) {
				$users_by_page_number[$j] = $users[$i];
				$j++;
			}
		}
		return $users_by_page_number;
	}
	
	/* This function will return an result object with the SQL result and LDAP result of all users
	 */
	public function get_user_from_username($username,$active = TRUE) {
		$ldap_conn = $this->prepare_ldap_conn();
    	$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_SEARCH_USERNAME, LDAP_ANON_SEARCH_PASSWORD);
		if($ldap_bind) {
			$fields = array('uid','cn','givenname','initials','sn','title','departmentnumber','o','physicaldeliveryofficename','telephonenumber','mobile');
			$base_dn = LDAP_ACCOUNT_GROUP;
			if(!$active) { $base_dn = LDAP_DISABLED_ACCOUNT_GROUP; }
			$search = ldap_search($ldap_conn, $base_dn, '(&(uid='.$username.'))', $fields);
			$entries = ldap_get_entries($ldap_conn, $search);
			$users = array();
			for($i = 0; $i < $entries['count']; $i++) {
				
				$entry = $entries[$i];
				$user = $this->get_db_user_from_username($entry['uid'][0])->result();
				
				if($user) {
					$user = $user[0]; //there should only be one result
					foreach($fields as $key) {
						if($key !== 'count' && isset($entry[$key][0])) { $user->$key = $entry[$key][0]; } else { $user->$key = FALSE; }
					}
					$user->groups = $this->get_user_groups($user->user_id);
					if($base_dn === LDAP_DISABLED_ACCOUNT_GROUP) { $user->disabled = TRUE; } else { $user->disabled = FALSE; }
					array_push($users,$user);
				}
				//else { return FALSE; }
			}
			if(count($users) === 1) { return $users[0]; } //if only one user, return just the one object
			return $users;
		}
		return FALSE;
	}
	
	/* This function will return an associative array with the SQL result and LDAP result of all disabled users
	 */
	public function get_disabled_users() {
		$ldap_conn = $this->prepare_ldap_conn();
    	$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_SEARCH_USERNAME, LDAP_ANON_SEARCH_PASSWORD);
		if($ldap_bind) {
			$fields = array('uid','cn','title','departmentnumber','o','physicaldeliveryofficename','telephonenumber','mobile');
			$search = ldap_search($ldap_conn, LDAP_DISABLED_ACCOUNT_GROUP, '(&(uid=*))', $fields);
			$entries = ldap_get_entries($ldap_conn, $search);
			$users = array();
			for($i = 0; $i < $entries['count']; $i++) {
				$entry = $entries[$i];
				$user = $this->get_db_user_from_username($entry['uid'][0])->result();
				if($user) {
					$user = $user[0]; //there should only be one result
					foreach($fields as $key) {
						if($key !== 'count' && isset($entry[$key][0])) { $user->$key = $entry[$key][0]; } else { $user->$key = FALSE; }
					}
					$user->groups = $this->get_user_groups($user->user_id);
					array_push($users,$user);
				}
				else { return FALSE; }
			}
		
			return $users;
		}
		return FALSE;
	}
	
	public function get_disabled_users_by_page_number($start, $size) {
		$users_by_page_number = array();
		$users = $this->get_disabled_users();
		if(!is_array($users)) { $users = array($users); }
		$j = 0;
		for($i = 0; $i < sizeOf($users); $i++) {
			
			if($i >= $start - 1 && $i < $start - 1 + $size) {
				$users_by_page_number[$j] = $users[$i];
				$j++;
			}
		}
		
		return $users_by_page_number;
	}
	
	/* This function returns an array of the common names of the groups a user belongs to (specified by
	 * user database id), this could be expanded to give more information on the groups.
	 */
	public function get_user_groups($user_id) {
		$username = $this->get_username_from_id($user_id);
		if(isset($username) && $username) {
			$dn = $this->get_dn_from_username($username);
			$ldap_conn = $this->prepare_ldap_conn();
			$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_SEARCH_USERNAME, LDAP_ANON_SEARCH_PASSWORD);
			if($ldap_bind) {
				$search = ldap_search($ldap_conn, LDAP_BASE_RDN, '(&(member='.$dn.'))', array('cn'));
				$entries = ldap_get_entries($ldap_conn, $search);
				$groups = array();
				for($i = 0; $i < $entries['count']; $i++) { array_push($groups,$entries[$i]['cn'][0]); }
				return $groups;
			}
		}
		return FALSE;
	}
	
   /*
    * This function will return an array of uids who are DaaS administrators
    */
    public function get_daas_admins(){
    	//connect to ldap
    	$ldap_conn = $this->prepare_ldap_conn();
    	$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_SEARCH_USERNAME, LDAP_ANON_SEARCH_PASSWORD);
    	$user_arr = array();
    	if($ldap_bind) {
    		//TODO: make this ou=api_admins configurable
    		//search for the api_admins group and pull all members
    		$search = ldap_search($ldap_conn, LDAP_BASE_RDN, '(&(ou=api_admins))', array('member'));
			$entries = ldap_get_entries($ldap_conn, $search);
			//grab the member entries
			$users = $entries[0]['member'];
			//loop through the members
			foreach ($users as $user){
				//grab the members that have a uid and return it
				if (preg_match("/uid=[^,]+/", $user, $uid) == true){  
					array_push($user_arr, substr($uid[0],4));  
				}
			}
    	}
    	return $user_arr;
    }
    
    /*
     * This function will return an array of uids who are DaaS administrators
    */
    public function get_app_admins($app_id){
    	//connect to ldap
    	$ldap_conn = $this->prepare_ldap_conn();
    	$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_SEARCH_USERNAME, LDAP_ANON_SEARCH_PASSWORD);
    	$user_arr = array();
    	
    	if($ldap_bind) {
    		//TODO: make this ou=api_admins configurable
    		//search for the api_admins group and pull all members
    		    		
    		$search = ldap_search($ldap_conn, LDAP_APPLICATION_GROUP, '(&(member=*))', array('member'));
    		$entries = ldap_get_entries($ldap_conn, $search);

    		foreach ($entries as $entry){
    			
    			$exploded_dn = ldap_explode_dn($entry['dn'],0); //explode dn so we can find parent dn
    			unset($exploded_dn['count'], $exploded_dn[0]); //remove values unncessary for parent dn
    			$parent_dn = implode(',',$exploded_dn); //impode array to get parent dn
    			
    			//get parent entry and grab the uid (which corresponds to the application's unique id in the database)
    			$parent_search = ldap_read($ldap_conn, $parent_dn, '(objectclass=*)', array('uid'));
    			$parent_entry = ldap_get_entries($ldap_conn, $parent_search);

    			if (isset($parent_entry[0]['uid'])){
    				$uid = (int)$parent_entry[0]['uid'][0];
    				
    				if ($uid == $app_id){
	    				$sec_search = ldap_read($ldap_conn, 'ou=admins,'.$parent_dn, '(&(ou=admins))', array('member'));
	    				$sec_entry = ldap_get_entries($ldap_conn, $sec_search);
	    				
	    				$users = $sec_entry[0]['member'];
	    				//loop through the members
	    				foreach ($users as $user){
	    					//grab the members that have a uid and return it
	    					if (preg_match("/uid=[^,]+/", $user, $uid) == true){
	    						array_push($user_arr, substr($uid[0],4));
	    					}
	    				}
	    				
    				}
    			}
    		}
    	}
    	
    	return $user_arr;
    }
    
    /*
     * This function will return an array of email addresses of app admins/pocs
     */
    public function get_request_emails($app_id){
    	$this->load->model('applicationrequestmodel');
    	$emails = array();
    	$request = $this->applicationrequestmodel->get_request($app_id);
    	if ($request){
    		$request_result = $request->result();
    		$requestor_id = $request_result[0]->requestor;
    		$requestor = $this->get_user_from_username($this->get_username_from_id($requestor_id));
    		array_push($emails, $request_result[0]->poc_email);
    		if($requestor){
    			$requestor_email = $requestor->user_ext_mail;
    			array_push($emails, $requestor_email);
    		}
    	}
    	
    	//remove any duplicates
    	$emails = array_unique($emails);
    	
    	return $emails;
    }
    
    /*
     * This function will return an array of email addresses of daas admins
     */
    public function get_emails_from_uid_array($admins){
    	$emails = array();
    	foreach($admins as $admin){
    		$query = $this->db->query('SELECT user_ext_mail FROM users WHERE username='.$this->db->escape($admin));
    		if ($query){
    			$email = $query->result();
    			array_push($emails,$email[0]->user_ext_mail);
    		}
    	}
    	return $emails;
    }
	
	/* This function creates a user account with the provided attributes
	 */
	public function create_user($username, $app_id, $id, $mail, $attributes) {
		$this->load->library('encrypt');
		$this->load->model('applicationmodel');
		$password = $this->random_password();
		$attributes['userPassword'] = $this->ssha256_encode($password);
    	$create_in_db = $this->db->query('INSERT INTO users (username, user_org_id, user_created_time, user_ext_mail, user_ep) VALUES ('.$this->db->escape($username).','.$this->db->escape($id).','.$this->db->escape(time()).','.$this->db->escape($mail).','.$this->db->escape($this->encrypt->encode($password)).')');
		if($create_in_db) {	
			$ldap_conn = $this->prepare_ldap_conn();
			/* TO-DO: It should be possible to allow API requests to pass a client cert through nginx, which will check for validity and set
			 * certain server variables. We should be able to check those variables, and if valid, look up the user from the cert then perform
			 * an LDAP bind as the actual user.
			 */
			 //TO-DO: re-examine LDAP permissions for FOC
			 //because this is exposed as an API function, we cannot get client cert to link to a user account
			//so use anonymous LDAP admin account and log that the request came from this application
			$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_ADMIN_USERNAME, LDAP_ANON_ADMIN_PASSWORD);
			if($ldap_bind) {
				$create_in_ldap = ldap_add($ldap_conn, $this->get_dn_from_username($username),$attributes);
				if(!$create_in_ldap) {
					//delete entry in database if ldap add fails, or ldap and the database will be out of sync
					$this->db->query('DELETE FROM users WHERE username='.$this->db->escape($username).' AND user_org_id='.$this->db->escape($id));
					return FALSE;
				}
				else { 
					//TO-DO: check here for success and delete other entries if this fails, then return FALSE
					$add_to_app = ldap_mod_add($ldap_conn,$this->applicationmodel->get_users_dn_from_app_id($app_id),array('member' => $this->get_dn_from_username($username)));
					return TRUE; 
				}
			}
		}
		return FALSE;
    }
	
	/* This function creates a user account with the provided attributes
	 * TO-DO: write corresponding Admin API function, if we decided apps will be able to update user settings
	 */
	public function update_user($user_id, $ext_mail, $attributes) {
		$user = $this->get_user($user_id);
		if($user) {
			$username = $user->username;
			$ldap_conn = $this->prepare_ldap_conn();
			//TO-DO: re-examine LDAP permissions for FOC
			//because this potentially will be exposed as an API function, we cannot get client cert to link to a user account
			//so use anonymous LDAP admin account and log that the request came from this application
			$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_ADMIN_USERNAME, LDAP_ANON_ADMIN_PASSWORD);
			if($ldap_bind) {
				$user_dn = $this->get_dn_from_username($username); 
				if($user->disabled) { $user_dn = $this->get_disabled_dn_from_username($username); }
				$result = array();
				foreach($attributes as $key => $attribute) {
					//if the attribute is blank, we can't set it to blank, we have to delete the entry
					if(strlen(trim($attribute)) > 0) { 
						$attribute_arr = array($key => $attribute);
						$result[$key] = ldap_modify($ldap_conn,$user_dn,$attribute_arr);
					}
					//suppress warnings because it will complain if it didn't exist and we tried to delete it again
					else {	
						$attribute_arr = array($key => array());
						$remove = @ldap_mod_del($ldap_conn,$user_dn,$attribute_arr); 
						//if the attribute didn't exist before (error 16), then we didn't need to get rid of it and therefore we did what we wanted
						if(!$remove && ldap_errno($ldap_conn) === 16) { $result[$key] = TRUE; }
						else { $result[$key] = $remove; }
					}
				}
				$result['ext_mail'] = $this->update_ext_mail($user->user_id,$ext_mail);
				return $result;
			}
		}
		return FALSE;
	}
	
	/* This function adds/removes the user specified by the user id (database index) to/from the group
	 * specified by the common name LDAP attribute given. 
	 * Add/Remove according to the specified action string (options 'add' or 'remove').
	 * TO-DO: add function in admin API for this?
	 */
	public function change_group_membership($action, $user_id, $group_cn) {
		if($action !== 'add' && $action !== 'remove') { return FALSE; }
		$username = $this->get_username_from_id($user_id);
		$user_dn = $this->get_dn_from_username($username); //even disabled users are stored in groups as their non-disabled dn
		
		$ldap_conn = $this->prepare_ldap_conn();
		//TO-DO: re-examine LDAP permissions for FOC
		//because this potentially will be exposed as an API function, we cannot get client cert to link to a user account
		//so use anonymous LDAP admin account and log that the request came from this application
		$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_ADMIN_USERNAME, LDAP_ANON_ADMIN_PASSWORD);
		if($ldap_bind) {
			$search = ldap_search($ldap_conn, LDAP_BASE_RDN, '(&(cn='.$group_cn.')(objectClass=groupOfNames))', array('dn'));
    		$entries = ldap_get_entries($ldap_conn, $search);
			if($entries['count'] === 1) { //has to be just one entry or it can't be sure what to remove
				$group_dn = $entries[0]['dn'];
				if($action === 'remove') { 
					$remove = @ldap_mod_del($ldap_conn,$group_dn,array('member'=>$user_dn));
					//if the attribute didn't exist before (error 16), then we didn't need to get rid of it and therefore we did what we wanted
					if(!$remove && ldap_errno($ldap_conn) === 16) { return TRUE; }
					else { return $remove; }
				}
				if($action === 'add') { 
					return ldap_mod_add($ldap_conn,$group_dn,array('member'=>$user_dn));
				}
			}
		}
		return FALSE;
	}
	
	/* This function will return a list of groups that the user with the permission set
	 * passed to this function as a parameter is allowed to add other users to, or remove them from.
	 */
	public function get_allowed_groups_for_access($permissions) {
		$ldap_conn = $this->prepare_ldap_conn();
		$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_SEARCH_USERNAME, LDAP_ANON_SEARCH_PASSWORD);
		if($ldap_bind) {
			//if the user is part of the API admins group, return all groups
			if($permissions['API']['admins']) {
				$search = ldap_search($ldap_conn, LDAP_BASE_RDN, '(objectClass=groupOfNames)', array('dn','cn'));
				$entries = ldap_get_entries($ldap_conn, $search);
				$result = array();
				foreach($entries as $key => $entry) {
					//don't include count array entry, and don't show groups that are for application entries only
					if(is_numeric($key) && $entry['dn'] !== LDAP_DIRECT_API_PERMISSIONS_GROUP && $entry['dn'] !== LDAP_ADMIN_API_PERMISSIONS_GROUP) { 
						array_push($result,$entry['cn'][0]); 
					}
				}
				return $result;
			}
			//TO-DO: for now, everyone else gets nothing, this will need to change
			return array();
		}
		return FALSE;
	}
	
	/* This function updates a user's external mail field in the database
	 */
	public function update_ext_mail($user_id, $ext_mail) {
		return $this->db->query('UPDATE users SET user_ext_mail='.$this->db->escape($ext_mail).' WHERE user_id='.$this->db->escape($user_id));
	}
	
	/* This function disables the user with the given user database id by
	 * placing it in a disabled users group in LDAP.
	 */ 
	public function disable_user($user_id) {
		$this->load->model('usersmodel');
		$username = $this->get_username_from_id($user_id);
		if($username) {
			$dn = $this->get_dn_from_username($username);
			if($dn) {
				$ldap_conn = $this->prepare_ldap_conn();
				$ldap_bind = $this->ldap_bind_current_user($ldap_conn); //logged in user must have privileges to disable
				if($ldap_bind) {
					return ldap_rename($ldap_conn, $dn,'uid=' . $username,LDAP_DISABLED_ACCOUNT_GROUP, TRUE);
				}
			}
		}
		return FALSE;
	}
	
	/* This function activates the user with the given user database id by
	 * placing it in a normal account group in LDAP.
	 */ 
	public function enable_user($user_id) {
		$username = $this->get_username_from_id($user_id);
		if($username) {
			$dn = $this->get_disabled_dn_from_username($username);
			if($dn) {
				$ldap_conn = $this->prepare_ldap_conn();
				//TO-DO: $ldap_bind = $this->ldap_bind_current_user($ldap_conn); //logged in user must have privileges to disable
				$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_ADMIN_USERNAME, LDAP_ANON_ADMIN_PASSWORD);
				if($ldap_bind) {
					return ldap_rename($ldap_conn, $dn,'uid=' . $username,LDAP_ACCOUNT_GROUP, TRUE);
				}
			}
		}
		return FALSE;
	}
	
	/* This function checks if a username exists. It returns TRUE if it exists, FALSE if it does not.
	 * It returns NULL on failure.
	 */
	public function username_exists($username) {
		$query = $this->db->query('SELECT username FROM users WHERE username='.$this->db->escape($username));
		if($query) {
			if($query->num_rows > 0) { return TRUE; }
			else { 
				$ldap_conn = $this->prepare_ldap_conn();
				$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_SEARCH_USERNAME, LDAP_ANON_SEARCH_PASSWORD);
				if($ldap_bind) {
					$search = ldap_search($ldap_conn, LDAP_GROUPS_GROUP, '(&(ou=' . $username . ')(cn=*))', array('ou'));
					$entries = ldap_get_entries($ldap_conn, $search);
					if($entries['count'] !== 0) { return TRUE; }
					else{
						$search = ldap_search($ldap_conn, LDAP_DISABLED_GROUPS_GROUP, '(&(ou=' . $username . ')(cn=*))', array('ou'));
						$entries = ldap_get_entries($ldap_conn, $search);
						if($entries['count'] !== 0) { return TRUE; }
						else{return FALSE;}
					}
				}
				else { return NULL; }
			}
		}
		else { return NULL; }
	}
	
	/* This function checks if an organizational id (EDIPI for DoD) is already in use.
	 * It returns NULL on failure.
	 */
	public function org_id_exists($id) {
		$id_query = $this->db->query('SELECT username FROM users WHERE user_org_id='.$this->db->escape($id));
		if($id_query->num_rows() === 0) { return FALSE; } //if no user records for organizational id
		else if($id_query->num_rows() === 1) { return TRUE; } //if no user records for organizational id
		else { return NULL; } //if too many user records for organizational id
	}
	
	/* This function checks if an organizational id (EDIPI for DoD) is already in use for a given application.
	 */
	public function org_id_linked($id,$application) {
		$id_query = $this->db->query('SELECT username FROM users WHERE user_org_id='.$this->db->escape($id));
		if($id_query->num_rows() === 0) { return FALSE; } //if no user records for organizational id
		if($id_query->num_rows() > 1) { return NULL; } //if too many user records for organizational id
		if($id_query) {
			$row = $id_query->row_array(0);
			$username = $row['username'];
			$ldap_conn = $this->prepare_ldap_conn();
			$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_SEARCH_USERNAME, LDAP_ANON_SEARCH_PASSWORD);
			if($ldap_bind) {
				$dn = $this->get_dn_from_username($username);
				//search for groups that this user is in
				$search = ldap_search($ldap_conn, LDAP_BASE_RDN, '(&(member=' . $dn . '))', array('dn'));
				$entries = ldap_get_entries($ldap_conn, $search);
				if($entries['count'] === 0) { return FALSE; } //if no entries, then it does not exist
				foreach($entries as $entry) {
					if(isset($entry['dn'])) {
						$exploded_dn = ldap_explode_dn($entry['dn'],0); //explode dn so we can find parent dn
						unset($exploded_dn['count'], $exploded_dn[0]); //remove values unncessary for parent dn
						$parent_dn = implode(',',$exploded_dn); //impode array to get parent dn
						
						//get parent entry and grab the uid (which corresponds to the application's unique id in the database)
						$parent_search = ldap_read($ldap_conn, $parent_dn, '(objectclass=*)', array('uid'));
						$parent_entry = ldap_get_entries($ldap_conn, $parent_search);
						if (isset($parent_entry[0]['uid']))
							$uid = (int)$parent_entry[0]['uid'][0];
						
						//get raw values of dn parts so we can check group name
						$exploded_dn_values = ldap_explode_dn($entry['dn'],1); 
						//if the uid matches the application we are checking and the user is in the application's user group
						if(($exploded_dn_values[0] === LDAP_APPLICATION_USER_GROUP_NAME) && ($uid === $application)) { return TRUE; }
					}
				}
				return FALSE;
			}
			else { return NULL; } //return NULL if status can't be determined due to ldap bind fail
		}
		else { return NULL; } //return NULL if status can't be determined due to query fail
	}
	
	/* This function returns a user id in the database that is associated with a provided direct address
	 */
	public function get_id_from_direct_address($address) {
		$ldap_conn = $this->prepare_ldap_conn();
		$ldap_bind = ldap_bind($ldap_conn, LDAP_ANON_SEARCH_USERNAME, LDAP_ANON_SEARCH_PASSWORD);
		if($ldap_bind) {
			//search for groups that this user is in
			$search = ldap_search($ldap_conn, LDAP_BASE_RDN, '(&(mail=' . $address . '))', array('dn','uid'));
			$entries = ldap_get_entries($ldap_conn, $search);
			if($entries['count'] === 0 || $entries['count'] > 1) { return FALSE; } //if no entries, or too many entries
			if(isset($entries[0]['uid']) && $entries[0]['uid']['count'] === 1) {
				return $this->get_id_from_username($entries[0]['uid'][0]);
			}
		}
		return FALSE;
	}
	
	/* This function gets the user record in the database associated with a username.
	 */
	public function get_db_user_from_username($username) {
		return $this->db->query('SELECT * FROM users WHERE username='.$this->db->escape($username));
	}
	
	/* This function gets the user id in the database associated with a username.
	 */
	public function get_id_from_username($username) {
		$query = $this->db->query('SELECT user_id FROM users WHERE username='.$this->db->escape($username));
		if($query) {
			if($query->num_rows === 1) {
				$row = $query->row_array(0);
				return $row['user_id'];
			}
		}
		return FALSE;
	}
	/* This function gets the user ids in the database associated with a username.
	 */
	public function get_ids_from_username($username) {
		return  $this->db->query('SELECT user_id FROM users WHERE username like '.$this->db->escape($username));
	}
	function get_orgs_from_username($username) {
		return  $this->db->query('SELECT user_org_id FROM users WHERE username like '.$this->db->escape($username));
	}
	
	/* This function gets the username in the database associated with a user id.
	 */
	public function get_username_from_id($user_id) {
		$query = $this->db->query('SELECT username FROM users WHERE user_id='.$this->db->escape($user_id));
		if($query) {
			if($query->num_rows === 1) {
				$row = $query->row_array(0);
				return $row['username'];
			}
		}
		return FALSE;
	}
	
	/* This function gets the user id in the database associated with a organizational identifier.
	 * For DoD, the EDIPI of the CAC will be used as the organizational identifier.
	 */
	public function get_user_id_from_org_id($org_id) {
		$query = $this->db->query('SELECT user_id FROM users WHERE user_org_id='.$this->db->escape($org_id));
		if($query) {
			if($query->num_rows === 1) {
				$row = $query->row_array(0);
				return $row['user_id'];
			}
		}
		return FALSE;
	}
	
	/* This function gets the organizational identifier associated with the database user id
	 */
	public function get_org_id_from_user_id($user_id) {
		$query = $this->db->query('SELECT user_org_id FROM users WHERE user_id='.$this->db->escape($user_id));
		if($query) {
			if($query->num_rows === 1) {
				$row = $query->row_array(0);
				return $row['user_org_id'];
			}
		}
		return FALSE;
	}
	
	/* This function gets the username in the database associated with a organizational identifier.
	 * For DoD, the EDIPI of the CAC will be used as the organizational identifier.
	 */
	public function get_username_from_org_id($id) {
		$query = $this->db->query('SELECT username FROM users WHERE user_org_id='.$this->db->escape($id));
		if($query) {
			if($query->num_rows === 1) {
				$row = $query->row_array(0);
				return $row['username'];
			}
		}
		return FALSE;
	}
	
	/* This function retrieves user permissions from LDAP based on group membership.
	 * Returns FALSE on failure to retrieve permissions.
	 */
	public function get_user_permissions($username) {
		$dn = $this->get_dn_from_username($username);
		//connect to LDAP
		$ldap_conn = $this->prepare_ldap_conn();
		//if we can't even get the user id (user doesn't exist / failure) don't bother hitting LDAP
		$user_id = $this->get_id_from_username($username);
		if(!$user_id) { return array('Registered'=>FALSE); }
		//otherwise get the user's password from the database to bind to LDAP
		$enc_pwd = $this->get_encrypted_password($user_id);
		$ldap_bind = @ldap_bind($ldap_conn, $this->get_dn_from_username($username), $this->encrypt->decode($enc_pwd));
		if($ldap_bind) { 
			//search for groups that this user is in (added !(mail=*) to filter out mail groups)
			$search = ldap_search($ldap_conn, LDAP_BASE_RDN, '(&(member=' . $dn . ')(!(mail=*)))', array('dn'));
			$entries = ldap_get_entries($ldap_conn, $search);
			//set default permissions to lowest possible
			$permissions = array('API'=>array('admins'=>FALSE),'Application'=>NULL,'Registered'=>TRUE);
			foreach($entries as $entry) {
				//if they are in the API admin group, set API admin permissions
				if($entry['dn'] === LDAP_API_ADMIN_GROUP) { 
					$permissions['API']['admins'] = TRUE;
				}
				//everything else will be an application group, so we will have to check the group's parent to see what app it is
				else { 
					if(isset($entry['dn'])) {
						$exploded_dn = ldap_explode_dn($entry['dn'],0); //explode dn so we can find parent dn
						unset($exploded_dn['count'], $exploded_dn[0]); //remove values unncessary for parent dn
						$parent_dn = implode(',',$exploded_dn); //impode array to get parent dn
						
						//get parent entry and grab the uid (which corresponds to the application's unique id in the database)
						$parent_search = ldap_read($ldap_conn, $parent_dn, '(objectclass=*)', array('uid'));
						$parent_entry = ldap_get_entries($ldap_conn, $parent_search);
						
						$uid = $parent_entry[0]['uid'][0];
						
						//get raw values of dn parts so we can set group name  in permissions
						$exploded_dn_values = ldap_explode_dn($entry['dn'],1); 
						$permissions['Application'][$uid][$exploded_dn_values[0]] = TRUE;
					}
				}
			}
			return $permissions;
		}
		return FALSE;
	}
	
	/* This function uses organizational id (for DOD this is EDIPI) to pull the user permissions,
	 * which return as an associative array split between 'API' and 'Application' permissions.
	 */
	public function get_user_permissions_from_org_id($id) {
		$username = $this->get_username_from_org_id($id);
		$permissions = $this->get_user_permissions($username);
		return $permissions;
	} 
	
	/* Simple function to format a dn for a given username on this system
	 * abstracted to a function so that it can be changed in only
	 * one place if the format ever changes.
	 */ 
	public function get_dn_from_username($username) {
		return 'uid='.$username.','.LDAP_ACCOUNT_GROUP;
	}
	
	/* Similar to get dn, but gets the dn as if the account were disabled */
	public function get_disabled_dn_from_username($username) {
		return 'uid='.$username.','.LDAP_DISABLED_ACCOUNT_GROUP;
	}
	
	/* This function performs an ldap bind on the given ldap connection using the user org id stored in session
	 */
	 public function ldap_bind_current_user($ldap_conn) {
		$this->load->library(array('session','encrypt'));
		$enc_current_user_org_id = $this->session->userdata('user_id');
		if(strlen($enc_current_user_org_id) > 0 && isset($enc_current_user_org_id)) {
			$current_user_org_id = $this->encrypt->decode($enc_current_user_org_id);
			$username = $this->get_username_from_org_id($current_user_org_id);
			$user_id = $this->get_id_from_username($username);
			$enc_pwd = $this->get_encrypted_password($user_id);
			$ldap_bind = ldap_bind($ldap_conn, $this->get_dn_from_username($username), $this->encrypt->decode($enc_pwd));
			return $ldap_bind;
		}
		return FALSE;
	 }
	 
	/* -----------------------------*
	 *  PRIVATE FUNCTIONS           *
	 * -----------------------------*/
	 
	 /* This function searches for the encrypted password of a user in the database given user id.
	  * Returns the encrypted password on success, FALSE on failure.
	  */
	 private function get_encrypted_password($id) {
		$query = $this->db->query('SELECT user_ep FROM users WHERE user_id='.$this->db->escape($id));
		if($query) {
			if($query->num_rows === 1) {
				$row = $query->row_array(0);
				return $row['user_ep'];
			}
		}
		return FALSE;
	 }
	 
	 /* This function prepares a connection to LDAP using the configured constants for the application
	  * and the LDAP options required for the connection. Returns FALSE on failure, LDAP connection resource
	  * on success.
	  */
	 private function prepare_ldap_conn() {
		$ldap_conn = ldap_connect(LDAP_HOSTNAME, LDAP_PORT);
		if(!ldap_set_option($ldap_conn, LDAP_OPT_PROTOCOL_VERSION, 3)) { return FALSE; } 
		if(!ldap_set_option($ldap_conn, LDAP_OPT_REFERRALS, 0)) { return FALSE; }
		return $ldap_conn;
	 }
	
	/* This function generates a 32 character random string for use as a password,
	 * utilizing a mix of lower and upper case, numbers, and special characters.
	 */
	private function random_password() {
		$chars = array("a","b","c","d","e","f","g","h","i","j","k","l","m","n","o","p","q","r","s","t","u","v","w","x","y","z","A","B","C","D","E","F","G","H","I","J","K","L","M","N","O","P","Q","R","S","T","U","V","W","X","Y","Z","1","2","3","4","5","6","7","8","9","0","!","@","#","$","%","^","&","*","(",")","{","}","?");
		$pass = "";
		for($i = 0; $i < 32; $i++) {
			$pass .= $chars[array_rand($chars)];
		}
		return $pass;
	}
	
	/* This function encodes the given text as a salted SHA string for use in OpenLDAP.
	 */
	private function ssha256_encode($text) {
		$salt = hash('sha256', openssl_random_pseudo_bytes(32));
		$hash = "{SSHA256}".base64_encode(pack("H*",hash('sha256',$text.$salt)).$salt);
		return $hash;
	}
}